;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Variable and Breed declarations ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

breeds [
  students
  androids
]

globals [
  COOPERATE
  DEFECT 
  
  number-of-plays
  strategy-list
  
  strategy-totals        ;; list of totals for selected strategies, for plotting
  strategy-totals-count  ;; number of times that strategy has been used
  defect-total           ;; total number of times turtles have defected
  cooperate-total        ;; total number of times turtles have cooperated
  
  ;; Shapes
  shape-names            ;; list of names of the non-sick shapes a client's turtle can have
  colors                 ;; list of colors used for clients' turtles
  color-names            ;; list of names of colors used for students' turtles
  used-shape-colors      ;; list of shape-color pairs that are in use
  max-possible-codes     ;; total number of unique shape/color combinations
]

turtles-own [
  score                  ;; my current score
  defect-now?            ;; what will I do this round?
  partner                ;; the who of my partner
  total-defects          ;; counts how many times the partner has defected, used for certain strategies
  selected-strategy      ;; string that contains  the user's selected strategy
  
  total                  ;; the total score of the turtle, following all plays
  
  play-history           ;; stores all of your moves in this play
  play-partner-history   ;; stores all of your partners moves in this play

  base-shape             ;; original shape of a turtle

  user-code              ;; students create custom strategies, which are stored here
  code-changed?          ;; is true when the user changes given strategies
]

students-own [
  user-id                ;; unique id, input by the client when they log in, to identify each student turtle
]

;;;;;;;;;;;;;;;;;;;;;
;; Setup Functions ;;
;;;;;;;;;;;;;;;;;;;;;

to startup
  hubnet-set-client-interface "COMPUTER" [ "clients/Prisoners Dilemma Client.nlogo" ]
  hubnet-reset
  setup
end

to setup
  ca 
  setup-shapes
  setup-vars
  setup-plots
end

to setup-plots
  set-current-plot "Strategies"
  set-current-plot-pen "0"
  plotxy 0 0
  plotxy 9 0
end

to setup-shapes
  ;; most of these are to handle shapes and colors for uniqueness of students
  
  set shape-names ["wide wedge" "square" "car" "thin wedge" "big boat" "pickup truck"
                   "nu" "uu" "circle" "butterfly" "wolf" "sheep" "lobster" "monster"
                   "moose" "bear" "teddy bear"]
  set colors      [ white gray brown yellow green lime turquoise
                    cyan sky blue violet ]
                    
  ;; adjust a few colors so they don't blend in with the red infection dot too much
  set colors lput (orange + 1) colors
  set colors lput (magenta + 0.5) colors
  set colors lput (pink + 2.5) colors
  set color-names [ "white" "gray" "brown" "yellow" "green" "lime" "turquoise"
                    "cyan" "sky" "blue" "violet" "orange" "magenta" "pink" ]
  set max-possible-codes (length colors * length shape-names)
  set used-shape-colors []
end

to setup-vars

  ;; these are constant, for use when students code their own strategies
  set COOPERATE false                                  
  set DEFECT true

  set number-of-plays 10

  ;; this is used for the androids, in interpreting their strategy
  set strategy-list [ "random" "cooperate" "defect" "go-by-majority" "tit-for-tat" 
                      "suspicious-tit-for-tat" "tit-for-two-tats" "pavlov" 
                      "unforgiving" "custom-strategy" ]
        
  set strategy-totals []
  set strategy-totals-count []
  foreach strategy-list [
    set strategy-totals fput 0 strategy-totals 
    set strategy-totals-count fput 0 strategy-totals-count 
  ]
  set defect-total 0
  set cooperate-total 0

end

to create-android-player
  create-custom-androids 1 [
    set total 0
    set partner nobody
    set base-shape "android"
    set shape base-shape
    set color blue
    setup-turtle-vars
    set selected-strategy random-one-of but-last strategy-list
    set label selected-strategy
    set-code
  ]
end

;; Places turtles in random locations again, ready to find new partners.
to rerun  
  ask turtles [
    setup-turtle-vars
    set total 0
    set score 3
  ]
end

to setup-turtle-vars  ;; turtle procedure
  set partner nobody
  set heading random 360
  set defect-now? false
  set play-history []
  set play-partner-history [] 
  set total-defects 0
  set size 1 
  set total (total + score)
  set score 3
  setxy random-float screen-size-x
        random-float screen-size-y
end

;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Runtime Procedures;;;
;;;;;;;;;;;;;;;;;;;;;;;;;

to play-n-times

  if (any? turtles) [  
    do-plots
    listen-clients
    find-partners

    every 0.3 [
      play-a-round 
    ]

    ask turtles with [ length play-history = number-of-plays ] [ 
      without-interruption [ 
        ifelse breed = students [ 
          ;; students may have changed strategy during the round, and this change can only be made afterwards
          if not code-changed? [ 
            let strategy-index position selected-strategy strategy-list
            set strategy-totals (replace-item strategy-index strategy-totals ((item strategy-index strategy-totals) + score)) 
            set strategy-totals-count (replace-item strategy-index strategy-totals-count ((item strategy-index strategy-totals-count) + 1))
          ]
        ]
        [
          let strategy-index position selected-strategy strategy-list 
          set strategy-totals (replace-item strategy-index strategy-totals ((item strategy-index strategy-totals) + score)) 
          set strategy-totals-count (replace-item strategy-index strategy-totals-count ((item strategy-index strategy-totals-count) + 1))
        ]
        setup-turtle-vars
        set shape base-shape
      ] 
    ]    
  ]  
end

to play-a-round  ;; determines the actions of turtles each turn
  ask turtles with [ partner != nobody ] [ 
    custom-strategy 
    get-payoff                 ;; after the strategies are determined, the results of the round are determined
    if breed = students 
      [ send-info-to-clients ]
  ]
end


;; test user strategy.
to custom-strategy ;; turtle procedure

  carefully [
    set defect-now? (run-result user-code)
  ] [  
    ifelse (breed = students) [
      hubnet-send user-id "Errors:" (error-message)
      show "bad strategy survived!"
    ] [
      show "Problem on android " + who + ": " + error-message
    ]
  ]
end

to find-partners
  ;;  In this example, only turtles that haven't found a partner can move around.
  listen-clients
  every 0.1 [
    ask turtles with [partner = nobody] [
      ;;  randomly move about the screen
      if breed != students [ 
        rt random-normal 0 20
      ]
      fd 0.5
    ]
  ]
  
  ;; Ask unpartnered turtles to check if they are on a patch with a turtle that 
  ;; also don't have a partner. Using without-interruption guarantees this will 
  ;; be done to completion by a single turtle before another turtle starts.
  ask turtles [
    without-interruption [
      if (partner = nobody) and (any? other-turtles-here with [partner = nobody]) [
        set partner random-one-of other-turtles-here with [partner = nobody]
        set partner-of partner self
      ]
    ]
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;     Some Helpful Functions    ;;;;;;;;;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to set-defect-shape
  ifelse defect-now? [ 
    set shape "sick " + base-shape 
  ] [ 
    set shape "good " + base-shape 
  ]
end

to-report test-strategy [ snippet ]
  carefully [
    let tester (run-result snippet)
    ifelse not (tester = false or tester = true) [ 
      hubnet-send user-id "Errors:" ("the output must be either true, or false") report false 
    ] [ 
      report true 
    ] 
  ] [
    hubnet-send user-id "Errors:" (error-message)
    report false
  ]
end

to set-code  ;; outputs the code to the input box, for students to see and modify
  if selected-strategy = "random"
    [ set user-code ("ifelse-value (random 2 = 0)\n  [DEFECT] \n[COOPERATE]") stop ]
  if selected-strategy = "cooperate"
    [ set user-code ("COOPERATE") stop ]
  if selected-strategy = "defect"
    [ set user-code ("DEFECT") stop ]
  if selected-strategy = "go-by-majority"
    [ set user-code ("ifelse-value (empty? play-history)\n  [COOPERATE] \n[ \n  ifelse-value (total-defects / (length play-history) > 0.5)\n    [DEFECT] \n  [COOPERATE] \n]") stop ]
  if selected-strategy = "tit-for-tat"
    [ set user-code ("ifelse-value (empty? play-history)\n  [COOPERATE] \n[ \n  ifelse-value (last play-partner-history = DEFECT)\n    [DEFECT]\n  [COOPERATE] \n]") stop ]
  if selected-strategy = "suspicious-tit-for-tat"
    [ set user-code ("ifelse-value (empty? play-history)\n  [DEFECT] \n[ \n  ifelse-value (last play-partner-history = DEFECT)\n    [DEFECT]\n  [COOPERATE] \n]") stop ]
  if selected-strategy = "tit-for-two-tats"
    [ set user-code ("ifelse-value (length play-history < 2 )\n  [COOPERATE] \n[ \n  ifelse-value ((last play-partner-history = DEFECT) and item (length play-partner-history - 2) play-partner-history = DEFECT)\n    [DEFECT] \n  [COOPERATE] \n]") stop ]
  if selected-strategy = "pavlov"
    [ set user-code ("ifelse-value (empty? play-history) \n[ \n  ifelse-value (random 2 = 0) [DEFECT] [COOPERATE] \n] \n[ \n  ifelse-value (last play-partner-history = DEFECT) \n  [\n    ifelse-value (last play-history = DEFECT)\n      [COOPERATE]\n    [DEFECT]\n  ]\n  [\n    ifelse-value (last play-history = DEFECT)\n      [DEFECT]\n    [COOPERATE]\n  ]\n]") stop ]
  if selected-strategy = "unforgiving"
    [ set user-code ("ifelse-value (empty? play-history)\n  [COOPERATE] \n[ \n  ifelse-value ((last play-partner-history = DEFECT) or (last play-history = DEFECT))\n    [DEFECT] \n  [COOPERATE] \n]") stop ]
end


;;;;;;;;;;;;;;;;;;;;;;
;;; End Strategies ;;;
;;;;;;;;;;;;;;;;;;;;;;

to get-payoff ;;Turtle Procedure
  ifelse defect-now?-of partner        ;; if the partner has defected
  [
    set play-partner-history lput true play-partner-history  ;; it is recorded in the history of his partner
    ifelse defect-now?                 ;; if this player has defected
    [
       set score score + D-D 
       set play-history lput true play-history  ;; it is recorded in this player's history
    ] 
    [
       set score score + C-D 
       set play-history lput false play-history
    ]
    set total-defects total-defects + 1  ;; used for go-by-majority strategy
    set defect-total defect-total + 1    ;; when detecting a partner's defecting, it is totaled for plotting
  ] 
  [
    set play-partner-history lput false play-partner-history
    ifelse defect-now? 
    [ 
      set score score + D-C 
      set play-history lput true play-history
    ] 
    [
      set score score + C-C 
      set play-history lput false play-history
    ]
    set cooperate-total cooperate-total + 1  ;;cooperates are totaled for plotting
  ]  
  set-defect-shape
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;            Plotting           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to do-plots  ;;firsts plots the average turnout for provided strategies, then the rest
  plot-strategies
  plot-cooperate-defect
end

to plot-strategies  ;;plots the average scores for each of the given strategies
  set-current-plot "Strategies"
    
  let i 0
  foreach (but-last strategy-list)
  [
    set-current-plot-pen ?
    if ((item i strategy-totals-count) != 0) 
    [ 
      plot-pen-reset
      set-plot-pen-mode 1
      plotxy i ( (item i strategy-totals) / (item i strategy-totals-count) )
    ]
    set i ( i + 1 )
  ]
end

to plot-cooperate-defect ;;plots the total number of times that turtles have cooperated or defected
  set-current-plot "C-D Plot"
  set-current-plot-pen "cooperate"
  plot cooperate-total
  set-current-plot-pen "defect"
  plot defect-total
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Code for interacting with the clients ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; determines which client sent a command, and what the command was
to listen-clients
  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    ifelse hubnet-enter-message?
    [ create-new-student ]
    [
      ifelse hubnet-exit-message?
      [ remove-student ]
      [ execute-command hubnet-message-tag ]
    ]
  ]
end

;; NetLogo knows what each student turtle is supposed to be
;; doing based on the tag sent by the node:
to execute-command [command]
  if command = "strategy-choice"  ;; the user wants to change his/her strategy, by selecting a pre-made strategy
  [ 
    ask students with [ user-id = hubnet-message-source ] 
    [ 
      ifelse (allow-strategy-change?) ;; if this is permitted under the current game-mode
      [
        hubnet-send user-id "Errors:" ("")
        set selected-strategy hubnet-message
        set-code
        hubnet-send user-id "strategy" user-code
        send-info-to-clients
        set code-changed? false
      ]
      [
        hubnet-send user-id "Errors:" ("You cannot change your strategy while playing")
        hubnet-send user-id "strategy-choice" (selected-strategy)
      ]
    ] 
  ]
  if command = "strategy"             ;; the user has modified the code of the current strategy
  [ 
     ask students with [ user-id = hubnet-message-source ] 
     [ 
       if (hubnet-message = user-code or not allow-strategy-change?) [ stop ]
       hubnet-send user-id "Errors:" ("")
       ;; code is taken, and tested for accuracy, so students can make more changes before playing another round
       if ( test-strategy hubnet-message )
       [ 
         set user-code hubnet-message  
         set code-changed? true
       ]
     ]
  ]
  if command = "Up"
    [ execute-move 0 ]
  if command = "Down"
    [ execute-move 180 ]
  if command = "Left"
    [ execute-move 270 ]
  if command = "Right"
    [ execute-move 90 ]
  if command = "Up-Left"
    [ execute-move 315 ]
  if command = "Up-Right"
    [ execute-move 45 ]
  if command = "Down-Left"
    [ execute-move 225 ]
  if command = "Down-Right"
    [ execute-move 135 ]
  if command = "Change Appearance"
    [ execute-change-turtle  ]
end

;; Create a turtle, set its shape, color, and position
;; and tell the node what its turtle looks like and where it is
to create-new-student
  create-custom-students 1
  [
    setup-student-vars
    hubnet-send user-id "strategy" user-code    
    send-info-to-clients
  ]
end

to execute-move [angle]
  ask students with [user-id = hubnet-message-source ]
  [
      set heading angle
  ]
end

;; sets the turtle variables to appropriate initial values
to setup-student-vars  ;; turtle procedure
  set user-id hubnet-message-source
  set total 0
  setup-turtle-vars
  set selected-strategy "random"
  set code-changed? false
  set-code
  set-unique-shape-and-color
end

;; pick a base-shape and color for the turtle
to set-unique-shape-and-color
  let code random max-possible-codes
  while [member? code used-shape-colors and count students < max-possible-codes]
  [
    set code random max-possible-codes
  ]
  set used-shape-colors (lput code used-shape-colors)
  set base-shape item (code mod length shape-names) shape-names
  set shape base-shape
  set color item (code / length shape-names) colors
end

;; report the string version of the turtle's color
to-report color-string [color-value]
  report item (position color-value colors) color-names
end

;; sends the appropriate monitor information back to the client
to send-info-to-clients
  hubnet-send user-id "You are a:" (color-string color) + " " + base-shape
  hubnet-send user-id "Your Score:" (score)
  hubnet-send user-id "Your Total:" (total) 
  ifelse partner != nobody
  [ 
    hubnet-send user-id "Partner's Score:" (score-of partner) 
    hubnet-send user-id "Partner's History:" (map [ ifelse-value (? = true) ["D "] ["C "] ] play-partner-history)
    hubnet-send user-id "Your History:" ( map [ ifelse-value (? = true) ["D "] ["C "] ] play-history)
    hubnet-send user-id "Points:" (map [ifelse-value ((?1 = false) and (?2 = false)) [C-C] [ifelse-value ((?1 = false) and (?2 = true)) [C-D] [ ifelse-value ((?1 = true) and (?2 = false)) [D-C] [D-D]]]] play-history play-partner-history)
  ]
  [ 
    hubnet-send user-id "Partner's Score:" ("") 
    hubnet-send user-id "Partner's History:" ("")
    hubnet-send user-id "Your History:" ("")
  ]

end

;; Kill the turtle, set its shape, color, and position
;; and tell the node what its turtle looks like and where it is
to remove-student
  ask students with [user-id = hubnet-message-source]
  [
    set used-shape-colors remove my-code used-shape-colors
    die
  ]
end

;; translates a student turtle's shape and color into a code
to-report my-code
  report (position base-shape shape-names) + (length shape-names) * (position color colors)
end

;; users might want to change their shape and color, so that they can find themselves more easily
to execute-change-turtle
  ask students with [user-id = hubnet-message-source]
  [
    set used-shape-colors remove my-code used-shape-colors
    show-turtle
    set-unique-shape-and-color
    hubnet-send user-id "You are a:" (color-string color) + " " + base-shape
  ]
end

; *** NetLogo Model Copyright Notice ***
;
; This activity and associated models and materials was created as part of the projects:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and 
; INTEGRATED SIMULATION AND MODELING ENVIRONMENT.
; These projects gratefully acknowledge the support of the 
; National Science Foundation (REPP & ROLE programs) -- grant numbers 
; REC #9814682 and REC-0126227.
;
; Copyright 2003 by Uri Wilensky. Updated 2004. All rights reserved.
;
; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from the copyright holders.
; Contact the copyright holders for appropriate licenses for redistribution 
; for profit.
;
; To refer to this model in academic publications, please use:
; Wilensky, U. (2003).  NetLogo HubNet Prisoners Dilemma model.
; http://ccl.northwestern.edu/netlogo/models/HubNetPrisonersDilemma.
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.
;
; In other publications, please use:
; Copyright 2003 Uri Wilensky.  All rights reserved.
; See http://ccl.northwestern.edu/netlogo/models/HubNetPrisonersDilemma
; for terms of use.
;
; *** End of NetLogo Model Copyright Notice ***
@#$#@#$#@
GRAPHICS-WINDOW
541
10
941
431
7
7
26.0
1
12
1
1
1
0

CC-WINDOW
5
456
950
551
Command Center

BUTTON
27
10
103
43
Setup
setup
NIL
1
T
OBSERVER
T
NIL

SWITCH
339
10
525
43
allow-strategy-change?
allow-strategy-change?
0
1
-1000

BUTTON
27
54
143
90
Create android
create-android-player
NIL
1
T
OBSERVER
T
A

BUTTON
110
10
187
43
Rerun
rerun
NIL
1
T
OBSERVER
T
NIL

BUTTON
194
10
287
43
Play
play-n-times
T
1
T
OBSERVER
T
NIL

TEXTBOX
340
91
467
133
Payoffs \n(prisoner-opponent)

SLIDER
340
137
432
170
C-C
C-C
-5
5
1
1
1
NIL

SLIDER
431
137
523
170
C-D
C-D
-5
5
-4
1
1
NIL

SLIDER
340
170
432
203
D-C
D-C
-5
5
5
1
1
NIL

SLIDER
431
170
523
203
D-D
D-D
-5
5
-3
1
1
NIL

PLOT
5
139
303
441
Strategies
NIL
NIL
-1.0
10.0
0.0
10.0
true
true
PENS
"random" 1.0 0 -16777216 true
"cooperate" 1.0 0 -6524078 true
"defect" 1.0 0 -16776961 true
"go-by-majority" 1.0 0 -44544 true
"tit-for-tat" 1.0 0 -16711681 true
"suspicious-tit-for-tat" 1.0 0 -65281 true
"tit-for-two-tats" 1.0 0 -16711936 true
"pavlov" 1.0 0 -256 true
"unforgiving" 1.0 0 -65536 true
"0" 1.0 0 -16777216 false

PLOT
325
242
518
442
C-D Plot
NIL
NIL
0.0
10.0
0.0
10.0
true
true
PENS
"cooperate" 1.0 0 -11352576 true
"defect" 1.0 0 -16776961 true

@#$#@#$#@
WHAT IS IT?
-----------
This model is a HubNet version of the Prisoner's Dilemma.  The Prisoner's Dilemma is a famous game-theory situation that models the costs and benefits of collaboration or treason between free agents where there is a struggle over some capital. The game has implications for a variety of social situations that involve negotiations in the absence of absolute trust between agents, such as in politics and economics. The game is a succession of interactions between agents, with each agent trying to maximize personal gains over all interactions. In this game, aggressive behavior is not necessarily the best strategy for maximizing personal gains. The rules of the game are such that agents are penalized when behaving aggressively in interacting with another aggressive agent, At each interaction between agents, each agent decides whether to 'cooperate' or 'defect.' These household terms in economics come from the following hypothetical situation.  

The Prisoner's Dilemma presents an interesting problem: if you and your accomplice in crime are imprisoned, then in order to minimize the overall jail time you should cooperate with your partner by remaining silent and not confessing.  However, you may choose to defect the unsaid "contract" with your partner by confessing.Let's assume you have confessed. If your partner does not confess you will go free.  If your partner confesses, you will go to jail for three years, much better than the five you would have earned had you refused to confess.  Unfortunately, your partner is in the same position.  Acting rationally in the short term, you will both be worse off. For more introductory information, please refer in the NetLogo models library to the PD BASIC model found in the PRISONER'S DILEMMA suite. Note also that in this model, the traditional year values, such as '3 years' or '5 years,' are given for modification.


HOW IT WORKS
------------
This model allows for many students within a classroom to challenge each other over a network.  Students will keep switching partners by moving their agents around on the screen and they can switch strategies. Students compete to see who has gained the maximum points over all interactions.

Moreover, this model allows students to gain a grasp of computer programming. An input box on the screen allows students to edit and author strategies. Following are the strategies that students can choose from (see the STRATEGY-CHOICE choice button):

STRATEGIES:
Random - randomly cooperate or defect
Cooperate - cooperate always
Defect - defect always
Go-by-Majority - Totals the number of times that the partner has defected. {against you or regardless of whom the partner had been playing against? Andrei, please choose and phrase}  If the majority of the time, up to that point, the partner has defected, defect.  Otherwise, cooperate.
Tit-for-Tat - If the opponent cooperates this round cooperate next round.  If the opponent defects this round, defect next round.  Initially cooperate. {again-- against me or anyone?}
Suspicious-Tit-for-Tat - The Tit-for-Tat strategy, except initially defects
Tit-for-Two-Tats - If the opponent cooperates this round cooperate next round.  If the opponent defects two rounds in a row, defect the next round.  Initially cooperate.
Pavlov - If the previous tactic (cooperate or defect) resulted in good {Andrei, what does this mean? Any positive point? The maximum possible?} points, stick with that tactic.  Otherwise, switch strategies.  Currently, "Success" happens when the partner cooperates, so this will keep switching if the opponent is always defecting.  Initially random.
Unforgiving - Cooperate always unless the opponent defects once.  Upon opponent defection retaliate by defecting always.


RULES FOR THE ITERATED PRISONER'S DILEMMA  
-----------------------------------------
1.  This game will consist of matches in which each student competes against one opponent at a time.  Students may search for partners on their own.  Once paired, students will play 10 rounds, and then separate and look for other partners.

2.  Each round the student and his/her partner will earn or lose points by either cooperating (C) or defecting (D) based on the rules set in their individual client. Each students' strategy is either selected (the STRATEGY-CHOICE ) or edited on the HubNet client. {Andrei, can players change their strat while in a given interaction?}

3.  The point system is determined by the administrator, using the 4 sliders under "PAYOFF." That is, the administrator does not change the basic game but the value of the prizes and penalties.


HOW TO USE IT
-------------

Buttons
-------
SETUP: Begin playing the iterated prisoner's dilemma. 
RERUN: All players set their total back to zero, and are ready to replay
PLAY: When students pair up, they will play a user-defined number of turns before separating
CREATE ANDROID: Creates a computer player to compete with students

Switches
--------
ALLOW-STRATEGY-CHANGE?: If on, students will be able to change their strategies. If off, they will be locked into the last working strategy in their client until it is turned on again.

Sliders
-------

COMPUTER STRATEGY - Select the computer's strategy from the drop-down list below.

Plots
-----
STRATEGY PLOTS: Shows the total points accumulated for each given strategy.  Once a strategy is modified by a student, the results of that student's success/failure will not be plotted on this.

C-D PLOT: Plots the total number of times players have cooperated, and the total number of times players have defected.


THINGS TO NOTICE
----------------

Watch the plots for the different strategies.  Is there a serious difference between tit-for-tat and suspicious-tit-fot-tat?  Does one strategy really beat out the rest?

See how often students cooperate and defect.  In the long run, do students learn to switch to more trusting strategies?  Or do students all too often defect?


THINGS TO TRY
-------------
1.  Experiment with playing different strategies one against the other.  Which strategies do the best?  Which do the worst?  Why?

2. Let it run for a while with hundreds of computer players to see which strategies win under these conditions.

3. There is a mode in which students cannot change their strategies.  Encourage them to code their own strategies, and then make them stick to that.  Which strategy seemed to work the best?


EXTENDING THE MODEL
-------------------
Students know the history of their last round, but not any of the previous rounds.  Think of a way to have clients be able to store the history of previous plays, and know what their current partner has done in other rounds.


NETLOGO FEATURES
----------------
Note the use of the turtle variable LABEL to display each turtle's average score in the GRAPHICS WINDOW.


RELATED MODELS
--------------
PD Basic, PD N-Person Iterated, PD Basic Evolutionary, PD 2-Person Iterated


CREDITS AND REFERENCES
----------------------
To refer to this model in academic publications, please use: Wilensky, U. (2004).  HubNet Prisoners Dilemma model. http://ccl.northwestern.edu/netlogo/models/HubNetPrisonersDilemma. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

@#$#@#$#@
default
false
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

android
false
0
Rectangle -7566196 true true 105 74 210 239
Polygon -7566196 true true 104 78 34 129 47 148 114 89
Polygon -7566196 true true 198 81 274 108 258 142 192 104
Polygon -7566196 true true 115 239 115 289 133 289 133 237
Polygon -7566196 true true 176 235 176 287 192 287 192 234
Rectangle -7566196 true true 119 12 194 73
Rectangle -16777216 true false 129 22 147 36
Rectangle -16777216 true false 164 23 184 37
Rectangle -16777216 true false 151 113 163 125
Rectangle -16777216 true false 153 142 164 154
Rectangle -16777216 true false 154 171 166 184

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

bear
false
0
Rectangle -7566196 true true 106 35 181 92
Circle -7566196 true true 110 17 18
Circle -7566196 true true 128 17 19
Rectangle -7566196 true true 80 58 111 89
Rectangle -7566196 true true 98 95 101 95
Rectangle -7566196 true true 92 93 192 104
Rectangle -7566196 true true 83 101 199 114
Rectangle -7566196 true true 78 107 205 130
Rectangle -7566196 true true 74 122 209 240
Rectangle -7566196 true true 77 236 204 249
Rectangle -7566196 true true 106 89 180 104
Rectangle -7566196 true true 80 247 198 258
Rectangle -7566196 true true 85 255 108 281
Rectangle -7566196 true true 169 252 195 283
Rectangle -7566196 true true 158 276 171 283
Rectangle -7566196 true true 74 276 90 281
Circle -7566196 true true 188 201 35
Circle -16777216 true false 111 40 13
Rectangle -7566196 true true 51 163 87 184
Rectangle -7566196 true true 45 166 53 169
Rectangle -7566196 true true 42 172 52 173
Rectangle -7566196 true true 38 178 53 183
Circle -7566196 true true 72 62 20

big boat
false
0
Polygon -6524078 true false 1 196 43 296 193 296 297 194
Rectangle -1 true false 135 14 149 194
Polygon -7566196 true true 151 14 173 18 193 30 211 48 239 88 251 118 271 170 271 184 253 176 227 170 199 172 177 180 161 190 165 160 169 122 165 78
Polygon -7566196 true true 133 36 115 50 77 86 47 122 7 152 33 156 57 164 77 178 91 188
Rectangle -7566196 true true 30 206 234 220
Rectangle -7566196 true true 52 224 234 236
Rectangle -7566196 true true 78 240 234 250

box
true
0
Polygon -7566196 true true 45 255 255 255 255 45 45 45

butterfly
false
0
Polygon -16777216 true false 151 76 138 91 138 284 150 296 162 286 162 91
Polygon -7566196 true true 164 106 184 79 205 61 236 48 259 53 279 86 287 119 289 158 278 177 256 182 164 181
Polygon -7566196 true true 136 110 119 82 110 71 85 61 59 48 36 56 17 88 6 115 2 147 15 178 134 178
Polygon -7566196 true true 46 181 28 227 50 255 77 273 112 283 135 274 135 180
Polygon -7566196 true true 165 185 254 184 272 224 255 251 236 267 191 283 164 276
Line -7566196 true 167 47 159 82
Line -7566196 true 136 47 145 81
Circle -7566196 true true 165 45 8
Circle -7566196 true true 134 45 6
Circle -7566196 true true 133 44 7
Circle -7566196 true true 133 43 8

car
false
15
Circle -1 false true 185 55 60
Circle -1 false true 183 186 61
Polygon -1 true true 214 52 214 22 182 26 162 38 144 74 138 102 100 120 99 161 102 201 118 246 152 267 190 275 210 251 187 240 178 200 204 182 215 181 214 118 193 112 182 98 181 72 198 52

circle
false
0
Circle -7566196 true true 30 30 240

computer
false
0
Rectangle -7566196 true true -55 -45 297 296
Rectangle -16777216 true false 43 27 264 263
Rectangle -11352576 true false 50 46 255 52
Rectangle -11352576 true false 53 238 250 246
Circle -11352576 true false 166 117 9
Rectangle -11352576 true false 153 70 205 76
Rectangle -11352576 true false 84 220 147 228

good android
false
0
Rectangle -7566196 true true 105 74 210 239
Polygon -7566196 true true 104 78 34 129 47 148 114 89
Polygon -7566196 true true 198 81 274 108 258 142 192 104
Polygon -7566196 true true 115 239 115 289 133 289 133 237
Polygon -7566196 true true 176 235 176 287 192 287 192 234
Rectangle -7566196 true true 119 12 194 73
Rectangle -16777216 true false 129 22 147 36
Rectangle -16777216 true false 164 23 184 37
Rectangle -16777216 true false 151 113 163 125
Rectangle -16777216 true false 153 142 164 154
Rectangle -16777216 true false 154 171 166 184
Circle -16711936 true false 4 125 112

good bear
false
0
Rectangle -7566196 true true 106 35 181 92
Circle -7566196 true true 110 17 18
Circle -7566196 true true 128 17 19
Rectangle -7566196 true true 80 58 111 89
Rectangle -7566196 true true 98 95 101 95
Rectangle -7566196 true true 92 93 192 104
Rectangle -7566196 true true 83 101 199 114
Rectangle -7566196 true true 78 107 205 130
Rectangle -7566196 true true 74 122 209 240
Rectangle -7566196 true true 77 236 204 249
Rectangle -7566196 true true 106 89 180 104
Rectangle -7566196 true true 80 247 198 258
Rectangle -7566196 true true 85 255 108 281
Rectangle -7566196 true true 169 252 195 283
Rectangle -7566196 true true 158 276 171 283
Rectangle -7566196 true true 74 276 90 281
Circle -7566196 true true 188 201 35
Circle -16777216 true false 111 40 13
Rectangle -7566196 true true 51 163 87 184
Rectangle -7566196 true true 45 166 53 169
Rectangle -7566196 true true 42 172 52 173
Rectangle -7566196 true true 38 178 53 183
Circle -7566196 true true 72 62 20
Circle -16711936 true false 57 88 136

good big boat
false
0
Polygon -6524078 true false 1 196 43 296 193 296 297 194
Rectangle -1 true false 135 14 149 194
Polygon -7566196 true true 151 14 173 18 193 30 211 48 239 88 251 118 271 170 271 184 253 176 227 170 199 172 177 180 161 190 165 160 169 122 165 78
Polygon -7566196 true true 133 36 115 50 77 86 47 122 7 152 33 156 57 164 77 178 91 188
Rectangle -7566196 true true 30 206 234 220
Rectangle -7566196 true true 52 224 234 236
Rectangle -7566196 true true 78 240 234 250
Circle -16711936 true false 1 134 123

good butterfly
false
0
Polygon -16777216 true false 151 76 138 91 138 284 150 296 162 286 162 91
Polygon -7566196 true true 164 106 184 79 205 61 236 48 259 53 279 86 287 119 289 158 278 177 256 182 164 181
Polygon -7566196 true true 136 110 119 82 110 71 85 61 59 48 36 56 17 88 6 115 2 147 15 178 134 178
Polygon -7566196 true true 46 181 28 227 50 255 77 273 112 283 135 274 135 180
Polygon -7566196 true true 165 185 254 184 272 224 255 251 236 267 191 283 164 276
Line -7566196 true 167 47 159 82
Line -7566196 true 136 47 145 81
Circle -7566196 true true 165 45 8
Circle -7566196 true true 134 45 6
Circle -7566196 true true 133 44 7
Circle -7566196 true true 133 43 8
Circle -16711936 true false 167 99 128

good car
false
15
Circle -1 false true 185 55 60
Circle -1 false true 183 186 61
Polygon -1 true true 214 52 214 22 182 26 162 38 144 74 138 102 100 120 99 161 102 201 118 246 152 267 190 275 210 251 187 240 178 200 204 182 215 181 214 118 193 112 182 98 181 72 198 52
Circle -16711936 true false 19 167 82

good circle
false
0
Circle -7566196 true true 35 35 230
Circle -16711936 true false 101 104 98

good lobster
false
0
Polygon -7566196 true true 105 106 122 238 100 259 100 276 118 275 150 240 180 275 197 275 197 258 174 238 196 105 178 77 123 77 105 106
Polygon -7566196 true true 112 103 84 78 105 60 94 14 85 57 78 19 67 59 76 83 110 118
Polygon -7566196 true true 192 120 228 85 241 57 231 22 223 59 217 17 205 61 216 79 192 108
Rectangle -7566196 true true 71 125 117 133
Rectangle -7566196 true true 73 149 119 157
Rectangle -7566196 true true 78 178 123 187
Rectangle -7566196 true true 183 131 219 138
Rectangle -7566196 true true 180 152 216 160
Rectangle -7566196 true true 173 176 217 184
Rectangle -7566196 true true 127 56 136 82
Rectangle -7566196 true true 160 55 169 81
Circle -16711936 true false 90 95 118

good monster
false
0
Rectangle -7566196 true true 77 100 213 245
Rectangle -7566196 true true 189 243 213 273
Rectangle -7566196 true true 78 242 102 271
Rectangle -7566196 true true 59 119 83 239
Rectangle -7566196 true true 203 118 225 239
Rectangle -7566196 true true 225 123 264 158
Rectangle -7566196 true true 262 124 272 131
Rectangle -7566196 true true 263 133 273 140
Rectangle -7566196 true true 263 145 271 156
Rectangle -7566196 true true 19 122 61 156
Rectangle -7566196 true true 9 125 19 130
Rectangle -7566196 true true 11 135 20 140
Rectangle -7566196 true true 9 146 20 151
Rectangle -7566196 true true 89 79 198 101
Rectangle -7566196 true true 112 35 176 82
Rectangle -16777216 true false 122 41 136 53
Rectangle -16777216 true false 149 39 167 53
Circle -1 true false 82 118 117
Polygon -16777216 true false 122 62 132 77 157 77 168 62
Polygon -1 true false 140 63 135 72 127 62
Polygon -1 true false 145 61 151 70 159 62
Polygon -7566196 true true 123 17 115 34 131 34
Polygon -7566196 true true 165 15 149 33 172 34
Circle -16711936 true false 77 102 140

good moose
false
0
Polygon -7566196 true true 226 121 90 121 60 136 61 181 106 195 210 195 255 181 255 135
Rectangle -7566196 true true 61 180 75 268
Rectangle -7566196 true true 89 187 104 275
Rectangle -7566196 true true 210 186 225 269
Rectangle -7566196 true true 241 178 254 275
Polygon -7566196 true true 62 138 60 107 23 107 17 102 18 85 24 82 54 79 66 63 87 64 94 128
Circle -16777216 true false 63 72 12
Polygon -7566196 true true 226 120 278 138 284 185 265 188 260 142
Polygon -7566196 true true 65 65 70 50
Polygon -7566196 true true 68 64 60 49 17 49 7 41 6 12 22 40 38 40 35 14 49 39 66 39 64 16 78 66 68 64
Polygon -7566196 true true 76 64 97 17 95 38 111 38 122 16 121 38 136 38 152 11 151 38 144 47 97 47 86 66
Circle -16711936 true false 79 87 115

good nu
false
0
Rectangle -7566196 true true 16 31 60 285
Rectangle -7566196 true true 120 31 164 285
Rectangle -7566196 true true 182 31 226 285
Rectangle -7566196 true true 256 31 298 285
Rectangle -7566196 true true 214 241 274 285
Polygon -7566196 true true 60 31 118 209 120 281 60 105
Rectangle -7566196 true true 110 211 140 249
Circle -16711936 true false 110 147 121

good pickup truck
false
0
Polygon -7566196 true true 285 208 285 178 279 164 261 144 229 132 217 132 213 106 199 84 171 68 149 68 129 68 129 148 1 148 1 156 19 164 19 222 285 222 283 174 283 176
Circle -16777216 true false 40 185 71
Circle -16777216 true false 192 191 66
Circle -7566196 true true 195 194 59
Circle -7566196 true true 43 188 64
Polygon -16777216 true false 197 94 149 94 157 128 209 128 205 112 203 102 197 94
Polygon -7566196 true true 21 142 139 142 139 136 13 136
Circle -16711936 true false 22 111 66

good sheep
false
14
Rectangle -1 true false 90 75 270 225
Circle -1 true false 15 75 150
Rectangle -16777216 true true 81 225 134 286
Rectangle -16777216 true true 180 225 238 285
Circle -16777216 true true 1 88 92
Circle -16711936 true false 112 46 144

good square
false
0
Rectangle -7566196 true true 45 45 255 255
Circle -16711936 true false 100 97 102

good teddy bear
false
0
Circle -7566196 true true 110 21 81
Circle -7566196 true true 97 7 39
Circle -7566196 true true 171 5 39
Polygon -7566196 true true 133 88 95 117 95 225 119 247 188 248 215 224 215 116 170 91
Circle -7566196 true true 134 238 34
Polygon -7566196 true true 197 113 249 88 261 91 268 106 262 116 205 139 197 113
Polygon -7566196 true true 115 116 66 90 54 93 45 110 50 117 103 145 115 116
Polygon -7566196 true true 104 204 54 233 54 244 63 257 71 256 117 227
Polygon -7566196 true true 194 228 240 255 248 254 260 238 257 231 204 207 194 228
Circle -1 true false 124 41 20
Circle -1 true false 158 42 20
Line -16777216 false 128 80 151 69
Line -16777216 false 151 69 174 83
Polygon -16711936 true false 151 208 111 173 132 153 151 169 171 154 189 169

good thin wedge
false
0
Polygon -7566196 true true 133 20 70 252 127 148 177 245
Circle -16711936 true false 165 140 87

good uu
false
0
Rectangle -7566196 true true 58 44 104 226
Rectangle -7566196 true true 88 178 194 224
Rectangle -7566196 true true 180 44 222 224
Rectangle -7566196 true true 122 74 168 284
Rectangle -7566196 true true 124 240 250 286
Rectangle -7566196 true true 240 76 284 286
Circle -16711936 true false 43 208 93

good wide wedge
false
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250
Circle -16711936 true false 202 65 90

good wolf
false
0
Rectangle -7566196 true true 15 105 105 165
Rectangle -7566196 true true 45 90 105 105
Polygon -7566196 true true 60 90 83 44 104 90
Polygon -16777216 true false 67 90 82 59 97 89
Rectangle -1 true false 48 93 59 105
Rectangle -16777216 true false 51 96 55 101
Rectangle -16777216 true false 0 121 15 135
Rectangle -16777216 true false 15 136 60 151
Polygon -1 true false 15 136 23 149 31 136
Polygon -1 true false 30 151 37 136 43 151
Rectangle -7566196 true true 105 120 263 195
Rectangle -7566196 true true 108 195 259 201
Rectangle -7566196 true true 114 201 252 210
Rectangle -7566196 true true 120 210 243 214
Rectangle -7566196 true true 115 114 255 120
Rectangle -7566196 true true 128 108 248 114
Rectangle -7566196 true true 150 105 225 108
Rectangle -7566196 true true 132 214 155 270
Rectangle -7566196 true true 110 260 132 270
Rectangle -7566196 true true 210 214 232 270
Rectangle -7566196 true true 189 260 210 270
Line -7566196 true 263 127 281 155
Line -7566196 true 281 155 281 192
Circle -16711936 true false 130 106 106

horse
false
0
Polygon -7566196 true true 226 121 90 121 60 136 61 181 106 195 210 195 255 181 255 135
Rectangle -7566196 true true 61 180 75 268
Rectangle -7566196 true true 89 187 104 275
Rectangle -7566196 true true 210 186 225 269
Rectangle -7566196 true true 241 178 254 275
Polygon -7566196 true true 62 138 60 107 23 107 17 102 18 85 24 82 54 79 66 63 87 64 94 128
Circle -16777216 true false 63 72 12
Polygon -7566196 true true 226 120 278 138 284 185 265 188 260 142
Polygon -7566196 true true 65 65 70 50
Polygon -7566196 true true 65 67 74 46 79 65 93 49 88 73 65 67

lobster
false
0
Polygon -7566196 true true 105 106 122 238 100 259 100 276 118 275 150 240 180 275 197 275 197 258 174 238 196 105 178 77 123 77 105 106
Polygon -7566196 true true 112 103 84 78 105 60 94 14 85 57 78 19 67 59 76 83 110 118
Polygon -7566196 true true 192 120 228 85 241 57 231 22 223 59 217 17 205 61 216 79 192 108
Rectangle -7566196 true true 71 125 117 133
Rectangle -7566196 true true 73 149 119 157
Rectangle -7566196 true true 78 178 123 187
Rectangle -7566196 true true 183 131 219 138
Rectangle -7566196 true true 180 152 216 160
Rectangle -7566196 true true 173 176 217 184
Rectangle -7566196 true true 127 56 136 82
Rectangle -7566196 true true 160 55 169 81

monster
false
0
Rectangle -7566196 true true 77 100 213 245
Rectangle -7566196 true true 189 243 213 273
Rectangle -7566196 true true 78 242 102 271
Rectangle -7566196 true true 59 119 83 239
Rectangle -7566196 true true 203 118 225 239
Rectangle -7566196 true true 225 123 264 158
Rectangle -7566196 true true 262 124 272 131
Rectangle -7566196 true true 263 133 273 140
Rectangle -7566196 true true 263 145 271 156
Rectangle -7566196 true true 19 122 61 156
Rectangle -7566196 true true 9 125 19 130
Rectangle -7566196 true true 11 135 20 140
Rectangle -7566196 true true 9 146 20 151
Rectangle -7566196 true true 89 79 198 101
Rectangle -7566196 true true 112 35 176 82
Rectangle -16777216 true false 122 41 136 53
Rectangle -16777216 true false 149 39 167 53
Circle -1 true false 82 118 117
Polygon -16777216 true false 122 62 132 77 157 77 168 62
Polygon -1 true false 140 63 135 72 127 62
Polygon -1 true false 145 61 151 70 159 62
Polygon -7566196 true true 123 17 115 34 131 34
Polygon -7566196 true true 165 15 149 33 172 34

moose
false
0
Polygon -7566196 true true 226 121 90 121 60 136 61 181 106 195 210 195 255 181 255 135
Rectangle -7566196 true true 61 180 75 268
Rectangle -7566196 true true 89 187 104 275
Rectangle -7566196 true true 210 186 225 269
Rectangle -7566196 true true 241 178 254 275
Polygon -7566196 true true 62 138 60 107 23 107 17 102 18 85 24 82 54 79 66 63 87 64 94 128
Circle -16777216 true false 63 72 12
Polygon -7566196 true true 226 120 278 138 284 185 265 188 260 142
Polygon -7566196 true true 65 65 70 50
Polygon -7566196 true true 68 64 60 49 17 49 7 41 6 12 22 40 38 40 35 14 49 39 66 39 64 16 78 66 68 64
Polygon -7566196 true true 76 64 97 17 95 38 111 38 122 16 121 38 136 38 152 11 151 38 144 47 97 47 86 66

nu
false
0
Rectangle -7566196 true true 16 31 60 285
Rectangle -7566196 true true 120 31 164 285
Rectangle -7566196 true true 182 31 226 285
Rectangle -7566196 true true 256 31 298 285
Rectangle -7566196 true true 214 241 274 285
Polygon -7566196 true true 60 31 118 209 120 281 60 105
Rectangle -7566196 true true 110 211 140 249

person
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

pickup truck
false
0
Polygon -7566196 true true 285 208 285 178 279 164 261 144 229 132 217 132 213 106 199 84 171 68 149 68 129 68 129 148 1 148 1 156 19 164 19 222 285 222 283 174 283 176
Circle -16777216 true false 40 185 71
Circle -16777216 true false 192 191 66
Circle -7566196 true true 195 194 59
Circle -7566196 true true 43 188 64
Polygon -16777216 true false 197 94 149 94 157 128 209 128 205 112 203 102 197 94
Polygon -7566196 true true 21 142 139 142 139 136 13 136

sheep
false
14
Rectangle -1 true false 90 75 270 225
Circle -1 true false 15 75 150
Rectangle -16777216 true true 81 225 134 286
Rectangle -16777216 true true 180 225 238 285
Circle -16777216 true true 1 88 92

sick android
false
0
Rectangle -7566196 true true 105 74 210 239
Polygon -7566196 true true 104 78 34 129 47 148 114 89
Polygon -7566196 true true 198 81 274 108 258 142 192 104
Polygon -7566196 true true 115 239 115 289 133 289 133 237
Polygon -7566196 true true 176 235 176 287 192 287 192 234
Rectangle -7566196 true true 119 12 194 73
Rectangle -16777216 true false 129 22 147 36
Rectangle -16777216 true false 164 23 184 37
Rectangle -16777216 true false 151 113 163 125
Rectangle -16777216 true false 153 142 164 154
Rectangle -16777216 true false 154 171 166 184
Circle -65536 true false 4 125 112

sick bear
false
0
Rectangle -7566196 true true 106 35 181 92
Circle -7566196 true true 110 17 18
Circle -7566196 true true 128 17 19
Rectangle -7566196 true true 80 58 111 89
Rectangle -7566196 true true 98 95 101 95
Rectangle -7566196 true true 92 93 192 104
Rectangle -7566196 true true 83 101 199 114
Rectangle -7566196 true true 78 107 205 130
Rectangle -7566196 true true 74 122 209 240
Rectangle -7566196 true true 77 236 204 249
Rectangle -7566196 true true 106 89 180 104
Rectangle -7566196 true true 80 247 198 258
Rectangle -7566196 true true 85 255 108 281
Rectangle -7566196 true true 169 252 195 283
Rectangle -7566196 true true 158 276 171 283
Rectangle -7566196 true true 74 276 90 281
Circle -7566196 true true 188 201 35
Circle -16777216 true false 111 40 13
Rectangle -7566196 true true 51 163 87 184
Rectangle -7566196 true true 45 166 53 169
Rectangle -7566196 true true 42 172 52 173
Rectangle -7566196 true true 38 178 53 183
Circle -7566196 true true 72 62 20
Circle -65536 true false 57 88 136

sick big boat
false
0
Polygon -6524078 true false 1 196 43 296 193 296 297 194
Rectangle -1 true false 135 14 149 194
Polygon -7566196 true true 151 14 173 18 193 30 211 48 239 88 251 118 271 170 271 184 253 176 227 170 199 172 177 180 161 190 165 160 169 122 165 78
Polygon -7566196 true true 133 36 115 50 77 86 47 122 7 152 33 156 57 164 77 178 91 188
Rectangle -7566196 true true 30 206 234 220
Rectangle -7566196 true true 52 224 234 236
Rectangle -7566196 true true 78 240 234 250
Circle -65536 true false 1 134 123

sick butterfly
false
0
Polygon -16777216 true false 151 76 138 91 138 284 150 296 162 286 162 91
Polygon -7566196 true true 164 106 184 79 205 61 236 48 259 53 279 86 287 119 289 158 278 177 256 182 164 181
Polygon -7566196 true true 136 110 119 82 110 71 85 61 59 48 36 56 17 88 6 115 2 147 15 178 134 178
Polygon -7566196 true true 46 181 28 227 50 255 77 273 112 283 135 274 135 180
Polygon -7566196 true true 165 185 254 184 272 224 255 251 236 267 191 283 164 276
Line -7566196 true 167 47 159 82
Line -7566196 true 136 47 145 81
Circle -7566196 true true 165 45 8
Circle -7566196 true true 134 45 6
Circle -7566196 true true 133 44 7
Circle -7566196 true true 133 43 8
Circle -65536 true false 167 99 128

sick car
false
15
Circle -1 false true 185 55 60
Circle -1 false true 183 186 61
Polygon -1 true true 214 52 214 22 182 26 162 38 144 74 138 102 100 120 99 161 102 201 118 246 152 267 190 275 210 251 187 240 178 200 204 182 215 181 214 118 193 112 182 98 181 72 198 52
Circle -65536 true false 19 167 82

sick circle
false
0
Circle -7566196 true true 35 35 230
Circle -65536 true false 101 104 98

sick lobster
false
0
Polygon -7566196 true true 105 106 122 238 100 259 100 276 118 275 150 240 180 275 197 275 197 258 174 238 196 105 178 77 123 77 105 106
Polygon -7566196 true true 112 103 84 78 105 60 94 14 85 57 78 19 67 59 76 83 110 118
Polygon -7566196 true true 192 120 228 85 241 57 231 22 223 59 217 17 205 61 216 79 192 108
Rectangle -7566196 true true 71 125 117 133
Rectangle -7566196 true true 73 149 119 157
Rectangle -7566196 true true 78 178 123 187
Rectangle -7566196 true true 183 131 219 138
Rectangle -7566196 true true 180 152 216 160
Rectangle -7566196 true true 173 176 217 184
Rectangle -7566196 true true 127 56 136 82
Rectangle -7566196 true true 160 55 169 81
Circle -65536 true false 90 95 118

sick monster
false
0
Rectangle -7566196 true true 77 100 213 245
Rectangle -7566196 true true 189 243 213 273
Rectangle -7566196 true true 78 242 102 271
Rectangle -7566196 true true 59 119 83 239
Rectangle -7566196 true true 203 118 225 239
Rectangle -7566196 true true 225 123 264 158
Rectangle -7566196 true true 262 124 272 131
Rectangle -7566196 true true 263 133 273 140
Rectangle -7566196 true true 263 145 271 156
Rectangle -7566196 true true 19 122 61 156
Rectangle -7566196 true true 9 125 19 130
Rectangle -7566196 true true 11 135 20 140
Rectangle -7566196 true true 9 146 20 151
Rectangle -7566196 true true 89 79 198 101
Rectangle -7566196 true true 112 35 176 82
Rectangle -16777216 true false 122 41 136 53
Rectangle -16777216 true false 149 39 167 53
Circle -1 true false 82 118 117
Polygon -16777216 true false 122 62 132 77 157 77 168 62
Polygon -1 true false 140 63 135 72 127 62
Polygon -1 true false 145 61 151 70 159 62
Polygon -7566196 true true 123 17 115 34 131 34
Polygon -7566196 true true 165 15 149 33 172 34
Circle -65536 true false 77 102 140

sick moose
false
0
Polygon -7566196 true true 226 121 90 121 60 136 61 181 106 195 210 195 255 181 255 135
Rectangle -7566196 true true 61 180 75 268
Rectangle -7566196 true true 89 187 104 275
Rectangle -7566196 true true 210 186 225 269
Rectangle -7566196 true true 241 178 254 275
Polygon -7566196 true true 62 138 60 107 23 107 17 102 18 85 24 82 54 79 66 63 87 64 94 128
Circle -16777216 true false 63 72 12
Polygon -7566196 true true 226 120 278 138 284 185 265 188 260 142
Polygon -7566196 true true 65 65 70 50
Polygon -7566196 true true 68 64 60 49 17 49 7 41 6 12 22 40 38 40 35 14 49 39 66 39 64 16 78 66 68 64
Polygon -7566196 true true 76 64 97 17 95 38 111 38 122 16 121 38 136 38 152 11 151 38 144 47 97 47 86 66
Circle -65536 true false 79 87 115

sick nu
false
0
Rectangle -7566196 true true 16 31 60 285
Rectangle -7566196 true true 120 31 164 285
Rectangle -7566196 true true 182 31 226 285
Rectangle -7566196 true true 256 31 298 285
Rectangle -7566196 true true 214 241 274 285
Polygon -7566196 true true 60 31 118 209 120 281 60 105
Rectangle -7566196 true true 110 211 140 249
Circle -65536 true false 110 147 121

sick pickup truck
false
0
Polygon -7566196 true true 285 208 285 178 279 164 261 144 229 132 217 132 213 106 199 84 171 68 149 68 129 68 129 148 1 148 1 156 19 164 19 222 285 222 283 174 283 176
Circle -16777216 true false 40 185 71
Circle -16777216 true false 192 191 66
Circle -7566196 true true 195 194 59
Circle -7566196 true true 43 188 64
Polygon -16777216 true false 197 94 149 94 157 128 209 128 205 112 203 102 197 94
Polygon -7566196 true true 21 142 139 142 139 136 13 136
Circle -65536 true false 22 111 66

sick sheep
false
14
Rectangle -1 true false 90 75 270 225
Circle -1 true false 15 75 150
Rectangle -16777216 true true 81 225 134 286
Rectangle -16777216 true true 180 225 238 285
Circle -16777216 true true 1 88 92
Circle -65536 true false 112 46 144

sick square
false
0
Rectangle -7566196 true true 45 45 255 255
Circle -65536 true false 100 97 102

sick teddy bear
false
0
Circle -7566196 true true 110 21 81
Circle -7566196 true true 97 7 39
Circle -7566196 true true 171 5 39
Polygon -7566196 true true 133 88 95 117 95 225 119 247 188 248 215 224 215 116 170 91
Circle -7566196 true true 134 238 34
Polygon -7566196 true true 197 113 249 88 261 91 268 106 262 116 205 139 197 113
Polygon -7566196 true true 115 116 66 90 54 93 45 110 50 117 103 145 115 116
Polygon -7566196 true true 104 204 54 233 54 244 63 257 71 256 117 227
Polygon -7566196 true true 194 228 240 255 248 254 260 238 257 231 204 207 194 228
Circle -1 true false 124 41 20
Circle -1 true false 158 42 20
Line -16777216 false 128 80 151 69
Line -16777216 false 151 69 174 83
Polygon -65536 true false 151 208 111 173 132 153 151 169 171 154 189 169

sick thin wedge
false
0
Polygon -7566196 true true 133 20 70 252 127 148 177 245
Circle -65536 true false 165 140 87

sick uu
false
0
Rectangle -7566196 true true 58 44 104 226
Rectangle -7566196 true true 88 178 194 224
Rectangle -7566196 true true 180 44 222 224
Rectangle -7566196 true true 122 74 168 284
Rectangle -7566196 true true 124 240 250 286
Rectangle -7566196 true true 240 76 284 286
Circle -65536 true false 43 208 93

sick wide wedge
false
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250
Circle -65536 true false 202 65 90

sick wolf
false
0
Rectangle -7566196 true true 15 105 105 165
Rectangle -7566196 true true 45 90 105 105
Polygon -7566196 true true 60 90 83 44 104 90
Polygon -16777216 true false 67 90 82 59 97 89
Rectangle -1 true false 48 93 59 105
Rectangle -16777216 true false 51 96 55 101
Rectangle -16777216 true false 0 121 15 135
Rectangle -16777216 true false 15 136 60 151
Polygon -1 true false 15 136 23 149 31 136
Polygon -1 true false 30 151 37 136 43 151
Rectangle -7566196 true true 105 120 263 195
Rectangle -7566196 true true 108 195 259 201
Rectangle -7566196 true true 114 201 252 210
Rectangle -7566196 true true 120 210 243 214
Rectangle -7566196 true true 115 114 255 120
Rectangle -7566196 true true 128 108 248 114
Rectangle -7566196 true true 150 105 225 108
Rectangle -7566196 true true 132 214 155 270
Rectangle -7566196 true true 110 260 132 270
Rectangle -7566196 true true 210 214 232 270
Rectangle -7566196 true true 189 260 210 270
Line -7566196 true 263 127 281 155
Line -7566196 true 281 155 281 192
Circle -65536 true false 130 106 106

square
false
0
Rectangle -7566196 true true 48 27 253 222

teddy bear
false
0
Circle -7566196 true true 110 21 81
Circle -7566196 true true 97 7 39
Circle -7566196 true true 171 5 39
Polygon -7566196 true true 133 88 95 117 95 225 119 247 188 248 215 224 215 116 170 91
Circle -7566196 true true 134 238 34
Polygon -7566196 true true 197 113 249 88 261 91 268 106 262 116 205 139 197 113
Polygon -7566196 true true 115 116 66 90 54 93 45 110 50 117 103 145 115 116
Polygon -7566196 true true 104 204 54 233 54 244 63 257 71 256 117 227
Polygon -7566196 true true 194 228 240 255 248 254 260 238 257 231 204 207 194 228
Circle -1 true false 124 41 20
Circle -1 true false 158 42 20
Line -16777216 false 127 75 150 85
Line -16777216 false 151 85 177 72
Polygon -1 true false 152 204 115 167 131 150 150 168 168 152 184 167

thin wedge
false
0
Polygon -7566196 true true 133 20 70 252 127 148 177 245

uu
false
0
Rectangle -7566196 true true 58 44 104 226
Rectangle -7566196 true true 88 178 194 224
Rectangle -7566196 true true 180 44 222 224
Rectangle -7566196 true true 122 74 168 284
Rectangle -7566196 true true 124 240 250 286
Rectangle -7566196 true true 240 76 284 286

wide wedge
false
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

wolf
false
0
Rectangle -7566196 true true 15 105 105 165
Rectangle -7566196 true true 45 90 105 105
Polygon -7566196 true true 60 90 83 44 104 90
Polygon -16777216 true false 67 90 82 59 97 89
Rectangle -1 true false 48 93 59 105
Rectangle -16777216 true false 51 96 55 101
Rectangle -16777216 true false 0 121 15 135
Rectangle -16777216 true false 15 136 60 151
Polygon -1 true false 15 136 23 149 31 136
Polygon -1 true false 30 151 37 136 43 151
Rectangle -7566196 true true 105 120 263 195
Rectangle -7566196 true true 108 195 259 201
Rectangle -7566196 true true 114 201 252 210
Rectangle -7566196 true true 120 210 243 214
Rectangle -7566196 true true 115 114 255 120
Rectangle -7566196 true true 128 108 248 114
Rectangle -7566196 true true 150 105 225 108
Rectangle -7566196 true true 132 214 155 270
Rectangle -7566196 true true 110 260 132 270
Rectangle -7566196 true true 210 214 232 270
Rectangle -7566196 true true 189 260 210 270
Line -7566196 true 263 127 281 155
Line -7566196 true 281 155 281 192

@#$#@#$#@
NetLogo 2.1beta4
@#$#@#$#@
setup
@#$#@#$#@
@#$#@#$#@
